﻿<?php

// Inicia a sessão e inclui arquivos de configuração.
session_start();
require_once 'conexao.php';
require_once 'verifica_login.php';

// Obtém os dados do usuário da sessão e os higieniza.
$usuario = $_SESSION['usuario'];
$usuario_id = intval($usuario['id']);
$nome = htmlspecialchars($usuario['nome'] ?? 'Professor');
$foto = htmlspecialchars($usuario['foto'] ?? 'sem-foto.png');
$mensagem_upload = $_SESSION['mensagem_upload'] ?? '';

// Limpa a mensagem da sessão para que ela não seja exibida novamente após a atualização da página
unset($_SESSION['mensagem_upload']);

// --- Processamento de Formulários (Requisições POST) ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Lógica para upload de nova foto de perfil.
    if (isset($_FILES['nova_foto']) && $_FILES['nova_foto']['error'] === UPLOAD_ERR_OK) {
        $diretorio_destino = 'fotos/';
        if (!is_dir($diretorio_destino)) {
            mkdir($diretorio_destino, 0777, true);
        }
        if (!is_writable($diretorio_destino)) {
            $_SESSION['mensagem_upload'] = "❌ Erro do servidor: o diretório de fotos não existe ou não tem permissão de escrita.";
            header("Location: painel_professor1.php");
            exit;
        }

        $extensao = strtolower(pathinfo($_FILES['nova_foto']['name'], PATHINFO_EXTENSION));
        $permitidas = ['jpg', 'jpeg', 'png', 'gif'];

        if (in_array($extensao, $permitidas)) {
            $novo_nome = 'prof_' . $usuario_id . '_' . time() . '.' . $extensao;
            $destino = $diretorio_destino . $novo_nome;

            if (move_uploaded_file($_FILES['nova_foto']['tmp_name'], $destino)) {
                $stmt = $conn->prepare("UPDATE usuarios SET foto = ? WHERE id = ?");
                $stmt->bind_param("si", $novo_nome, $usuario_id);
                if ($stmt->execute()) {
                    $_SESSION['usuario']['foto'] = $novo_nome;
                    $foto = $novo_nome;
                    $_SESSION['mensagem_upload'] = "✅ Foto de perfil atualizada com sucesso!";
                } else {
                    $_SESSION['mensagem_upload'] = "❌ Erro ao atualizar o banco de dados.";
                    unlink($destino); // Remove a foto se o DB falhar
                }
                $stmt->close();
            } else {
                $_SESSION['mensagem_upload'] = "❌ Erro ao fazer o upload da foto.";
            }
        } else {
            $_SESSION['mensagem_upload'] = "⚠️ Formato de imagem não permitido. Use JPG, PNG ou GIF.";
        }
    }
    // Não há mais lógica de chat aqui, então o redirecionamento é simplificado.
    header("Location: " . $_SERVER['PHP_SELF']);
    exit;
}

// --- Busca de Dados para Exibição ---

// Busca os últimos 5 avisos da administração.
$avisos = [];
$res_avisos = $conn->query("SELECT titulo, conteudo, data_publicacao FROM avisos_professores ORDER BY data_publicacao DESC LIMIT 5");
if ($res_avisos) {
    while ($row = $res_avisos->fetch_assoc()) {
        $avisos[] = $row;
    }
    $res_avisos->close();
}

// CORREÇÃO AQUI: Busca as turmas atribuídas ao professor, usando as colunas diretas da tabela 'turmas'
$turmas_professor = [];
$query_turmas = "
    SELECT DISTINCT 
        t.id, 
        t.nome AS nome_turma, 
        t.classe,    /* Coluna 'classe' diretamente da tabela turmas */
        t.curso,     /* Coluna 'curso' diretamente da tabela turmas */
        t.turno      /* Coluna 'turno' diretamente da tabela turmas */
    FROM turmas t 
    JOIN atribuicoes at ON t.id = at.turma_id
    WHERE at.professor_id = ?"; 

$stmt_turmas = $conn->prepare($query_turmas);
$stmt_turmas->bind_param("i", $usuario_id);
$stmt_turmas->execute();
$result_turmas = $stmt_turmas->get_result();
while ($row = $result_turmas->fetch_assoc()) {
    // Formata o nome completo da turma no formato 'classe-turma-curso-turno'
    // Adicione validação para garantir que os campos não são nulos ou vazios, se necessário
    $partes_turma = [];
    if (!empty($row['classe'])) {
        $partes_turma[] = htmlspecialchars($row['classe']);
    }
    if (!empty($row['nome_turma'])) {
        $partes_turma[] = htmlspecialchars($row['nome_turma']);
    }
    if (!empty($row['curso'])) {
        $partes_turma[] = htmlspecialchars($row['curso']);
    }
    if (!empty($row['turno'])) {
        $partes_turma[] = htmlspecialchars($row['turno']);
    }
    
    $row['nome_completo'] = implode('-', $partes_turma);
    // Se, por algum motivo, todas as partes estiverem vazias, defina um fallback
    if (empty($row['nome_completo'])) {
        $row['nome_completo'] = 'Turma Desconhecida (ID: ' . $row['id'] . ')';
    }

    $turmas_professor[] = $row;
}
$stmt_turmas->close();

// Busca dados para o gráfico de evolução de notas das turmas atribuídas
$labels_notas_turmas = ['1º Trimestre', '2º Trimestre', '3º Trimestre'];
$datasets_notas_turmas = [];
$colors_chart = ['#007bff', '#28a745', '#ffc107', '#dc3545', '#6f42c1', '#17a2b8', '#fd7e14', '#e83e8c'];
$color_index = 0;

foreach ($turmas_professor as $turma) {
    $turma_id = $turma['id'];
    // Usar o nome_completo para o label do gráfico
    $turma_nome_completo = $turma['nome_completo'];

    $query_notas_turma = "SELECT AVG(n.mt1) as media_mt1, AVG(n.mt2) as media_mt2, AVG(n.mt3) as media_mt3
                          FROM notas n
                          JOIN alunos a ON n.aluno_id = a.id
                          WHERE a.turma_id = ?
                          GROUP BY a.turma_id";
    $stmt_notas_turma = $conn->prepare($query_notas_turma);
    $stmt_notas_turma->bind_param("i", $turma_id);
    $stmt_notas_turma->execute();
    $result_notas_turma = $stmt_notas_turma->get_result();
    
    $medias = ['media_mt1' => 0, 'media_mt2' => 0, 'media_mt3' => 0];
    if ($result_notas_turma->num_rows > 0) {
        $medias = $result_notas_turma->fetch_assoc();
    }
    $stmt_notas_turma->close();

    $datasets_notas_turmas[] = [
        'label' => $turma_nome_completo, // Usando o nome_completo aqui
        'data' => [(float)$medias['media_mt1'], (float)$medias['media_mt2'], (float)$medias['media_mt3']],
        'backgroundColor' => $colors_chart[$color_index % count($colors_chart)],
        'borderColor' => $colors_chart[$color_index % count($colors_chart)],
        'borderWidth' => 1
    ];
    $color_index++;
}

// Frases do dia (código mantido, pois já estava funcional).
$frases_do_dia = [
    "Educar é semear com sabedoria e colher com paciência. 🌱",
    "A educação é a arma mais poderosa que você pode usar para mudar o mundo. - Nelson Mandela",
    "A mente não é um vaso a ser enchido, mas um fogo a ser aceso. - Plutarco",
    "O professor medíocre conta. O bom professor explica. O professor superior demonstra. O grande professor inspira. - William Arthur Ward",
    "Educar é impregnar de sentido o que fazemos a cada instante. - Paulo Freire",
    "A persistência é o caminho do êxito. - Charles Chaplin",
    "O único lugar onde o sucesso vem antes do trabalho é no dicionário. - Vidal Sassoon",
    "Aprender é a única coisa que a mente nunca se cansa, nunca tem medo e nunca se arrepende. - Leonardo da Vinci",
    "Não espere por oportunidades extraordinárias. Agarre ocasiões comuns e torne-as grandes. - Orison Swett Marden",
    "A paciência e a persistência têm um efeito mágico, diante do qual as dificuldades desaparecem e os obstáculos somem. - John Quincy Adams",
    "O sucesso é a soma de pequenos esforços repetidos dia após dia. - Robert Collier",
    "Onde quer que você vá, vá com todo o seu coração. - Confúcio",
    "A curiosidade é a lâmpada do conhecimento. - Provérbio Chinês",
    "O futuro pertence àqueles que acreditam na beleza de seus sonhos. - Eleanor Roosevelt",
    "A melhor maneira de prever o futuro é criá-lo. - Peter Drucker",
    "Aquele que passa por esta vida e não deixa a marca de si, não viveu. - Olavo Bilac",
    "Ensine as crianças, e não será necessário punir os homens. - Pitágoras",
    "Você não pode ensinar nada a um homem; você pode apenas ajudá-lo a encontrar a resposta dentro dele mesmo. - Galileu Galilei",
    "O conhecimento nos torna livres e a liberdade nos dá poder para criar um futuro melhor. - Anônimo",
    "A verdadeira educação consiste em tirar o melhor de si mesmo. - Mahatma Gandhi",
    "Cada aluno é uma semente que espera o momento certo para florescer. - Anônimo",
    "Sua maior força está em sua capacidade de inspirar e guiar os outros. - Anônimo",
    "A educação é a passagem da escuridão para a luz. - Allan Bloom",
    "A tarefa do educador é ser o guia que ilumina o caminho do aprendizado. - Anônimo",
    "Grandes mentes discutem ideias; mentes medianas discutem eventos; mentes pequenas discutem pessoas. - Eleanor Roosevelt",
    "A cada dia, você constrói o futuro de alguém com seu conhecimento e dedicação. - Anônimo",
    "A maior glória em viver não está em nunca cair, mas em levantar a cada queda. - Nelson Mandela",
    "O tempo é um bom professor, mas infelizmente ele mata todos os seus alunos. - Hector Berlioz",
    "Nunca pare de aprender, pois a vida nunca para de ensinar. - Anônimo",
    "O educador tem a chave que abre a porta do conhecimento para o mundo. - Anônimo",
    "O verdadeiro sucesso é a jornada, não o destino. - Zig Ziglar"
];
mt_srand(date('z'));
$frase_selecionada = $frases_do_dia[mt_rand(0, count($frases_do_dia) - 1)];

// Fecha a conexão com o banco de dados.
$conn->close();
?>
<!DOCTYPE html>
<html lang="pt">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Painel do Professor</title>
    <link rel="stylesheet" href="css/all.min.css">
<link rel="stylesheet" href="css/inter.css">
<script src="js/tailwindcss.js"></script> <script src="js/chart.umd.min.js"></script>
    <style>
        body { font-family: 'Inter', sans-serif; }
        .sidebar {
            transition: transform 0.3s ease-in-out;
            transform: translateX(-100%);
            z-index: 999;
        }
        .sidebar.active { transform: translateX(0); }
        .overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            z-index: 998;
            display: none;
        }
        .overlay.active { display: block; }
        @media (min-width: 768px) {
            .sidebar { transform: translateX(0); }
            .content { margin-left: 16rem; }
            .overlay { display: none !important; }
        }

        /* Estilo da mensagem de sucesso/erro */
        .mensagem {
            margin-top: 10px;
            font-weight: bold;
            padding: 10px;
            border-radius: 5px;
            text-align: center;
            margin-bottom: 20px;
        }
        .mensagem.sucesso {
            color: #28a745;
            background-color: #d4edda;
            border: 1px solid #c3e6cb;
        }
        .mensagem.erro {
            color: #dc3545;
            background-color: #f8d7da;
            border: 1px solid #f5c6cb;
        }
        .mensagem.aviso {
            color: #ffc107;
            background-color: #fff3cd;
            border: 1px solid #ffeeba;
        }
    </style>
</head>
<body class="bg-gray-100 min-h-screen flex flex-col">

    <header class="fixed top-0 left-0 w-full md:left-64 md:w-[calc(100%-16rem)] h-16 bg-blue-900 text-white flex items-center justify-between p-4 shadow-lg z-50">
        <button id="menu-toggle" class="md:hidden text-2xl focus:outline-none">
            <i class="fas fa-bars"></i>
        </button>
        <div class="flex items-center gap-4 ml-auto">
            <span class="hidden sm:block font-bold"><?= $nome ?></span>
            <img src="fotos/<?= $foto ?>" alt="Foto do Professor" class="w-10 h-10 rounded-full border-2 border-white object-cover">
        </div>
    </header>

    <div id="overlay" class="overlay"></div>

    <nav id="sidebar" class="sidebar fixed top-0 left-0 w-64 h-full bg-blue-900 text-white p-4 pt-16 shadow-lg overflow-y-auto">
        <div class="flex flex-col items-center justify-center p-4">
            <img src="fotos/<?= $foto ?>" alt="Foto do Professor" class="w-20 h-20 rounded-full border-4 border-white mb-2 object-cover">
            <span class="text-xl font-bold text-center"><?= $nome ?></span>
            <form method="post" enctype="multipart/form-data" class="mt-4 w-full text-center">
                <input type="file" name="nova_foto" id="nova_foto_upload" accept="image/*" class="hidden" onchange="this.form.submit()">
                <button type="button" onclick="document.getElementById('nova_foto_upload').click()" class="bg-blue-700 hover:bg-blue-800 text-white font-semibold py-2 px-4 rounded-full transition-colors duration-200">
                    Alterar Foto
                </button>
            </form>
        </div>
        <ul class="mt-8 space-y-2">
            <li><a href="mini_pauta_professor1.php" class="flex items-center gap-3 p-3 rounded-lg hover:bg-blue-700 transition duration-200"><i class="fas fa-pen"></i><span>Mini Pauta</span></a></li>

            <li><a href="caderneta_professor.php" class="flex items-center gap-3 p-3 rounded-lg hover:bg-blue-700 transition duration-200"><i class="fas fa-pen"></i><span>Caderneta</span></a></li>
             <li><a href="estatistica_professor.php" class="flex items-center gap-3 p-3 rounded-lg hover:bg-blue-700 transition duration-200"><i class="fas fa-pen"></i><span>Minha Estatística</span></a></li>

            <li><a href="alterar_senha.php" class="flex items-center gap-3 p-3 rounded-lg hover:bg-blue-700 transition duration-200"><i class="fas fa-key"></i><span>Alterar Senha</span></a></li>
            <li><a href="documentos_recebidos.php" class="flex items-center gap-3 p-3 rounded-lg hover:bg-blue-700 transition duration-200"><i class="fas fa-folder"></i><span>Documentos Recebidos</span></a></li>
            <li><a href="enviar_documento.php" class="flex items-center gap-3 p-3 rounded-lg hover:bg-blue-700 transition duration-200"><i class="fas fa-upload"></i><span>Enviar Documento</span></a></li>
            <li><a href="cadastrar_trabalho.php" class="flex items-center gap-3 p-3 rounded-lg hover:bg-blue-700 transition duration-200"><i class="fas fa-clipboard-list"></i><span>Cadastrar Trabalho</span></a></li>
            <li><a href="planificacao_modelo.php" class="flex items-center gap-3 p-3 rounded-lg hover:bg-blue-700 transition duration-200"><i class="fas fa-book"></i><span>Modelo de Planificação</span></a></li>
            <li><a href="grafico_notas.php" class="flex items-center gap-3 p-3 rounded-lg hover:bg-blue-700 transition duration-200"><i class="fas fa-chart-line"></i><span>Ver Gráfico</span></a></li>
            <li><a href="avaliacao_pedagogica.php" class="flex items-center gap-3 p-3 rounded-lg hover:bg-blue-700 transition duration-200"><i class="fas fa-poll"></i><span>Avaliação Pedagógica</span></a></li>
            <li><a href="historico_lancamentos.php" class="flex items-center gap-3 p-3 rounded-lg hover:bg-blue-700 transition duration-200"><i class="fas fa-history"></i><span>Histórico de Lançamentos</span></a></li>
            <li><a href="emitir_sugestao.php" class="flex items-center gap-3 p-3 rounded-lg hover:bg-blue-700 transition duration-200"><i class="fas fa-lightbulb"></i><span>Emitir Sugestão</span></a></li>
            <li><a href="ver_catedras.php" class="flex items-center gap-3 p-3 rounded-lg hover:bg-blue-700 transition duration-200"><i class="fas fa-university"></i><span>Cátedras</span></a></li>
            <li><a href="upload_catedra.php" class="flex items-center gap-3 p-3 rounded-lg hover:bg-blue-700 transition duration-200"><i class="fas fa-upload"></i><span>Upload Cátedra</span></a></li>
            <li><a href="login.php" class="flex items-center gap-3 p-3 rounded-lg hover:bg-blue-700 transition duration-200"><i class="fas fa-sign-out-alt"></i><span>Sair</span></a></li>
        </ul>
    </nav>

    <main class="content flex-grow p-5 md:ml-64 mt-16">
        <h1 class="text-3xl font-bold text-gray-800 mb-6">Início</h1>

        <?php if (!empty($mensagem_upload)): ?>
            <div class="mensagem <?= strpos($mensagem_upload, '✅') !== false ? 'sucesso' : (strpos($mensagem_upload, '❌') !== false ? 'erro' : 'aviso') ?>">
                <?= $mensagem_upload ?>
            </div>
        <?php endif; ?>

        <section class="bg-white p-6 rounded-xl shadow-md mb-6">
            <h2 class="text-xl font-semibold text-gray-700 mb-4">📚 Minhas Turmas Atribuídas</h2>
            <?php if (empty($turmas_professor)): ?>
                <p class="text-gray-600">Nenhuma turma atribuída a você.</p>
            <?php else: ?>
                <ul class="list-disc list-inside text-gray-700 space-y-1">
                    <?php foreach ($turmas_professor as $turma): ?>
                        <li><?= htmlspecialchars($turma['nome_completo']) ?></li>
                    <?php endforeach; ?>
                </ul>
            <?php endif; ?>
        </section>

        <section class="bg-white p-6 rounded-xl shadow-md mb-6">
            <h2 class="text-xl font-semibold text-gray-700 mb-4">📊 Evolução de Notas por Turma Atribuída</h2>
            <div class="relative h-96">
                <canvas id="graficoNotasProfessor"></canvas>
            </div>
        </section>

        <section class="bg-white p-6 rounded-xl shadow-md mb-6">
            <h2 class="text-xl font-semibold text-gray-700 mb-4">📢 Avisos da Administração</h2>
            <?php if (empty($avisos)): ?>
                <p class="text-gray-600">Nenhum aviso publicado até o momento.</p>
            <?php else: ?>
                <?php foreach ($avisos as $aviso): ?>
                    <div class="mb-4 border-l-4 border-blue-900 pl-4">
                        <strong class="text-gray-800"><?= htmlspecialchars($aviso['titulo']) ?></strong><br>
                        <small class="text-gray-500">Publicado em <?= date('d/m/Y', strtotime($aviso['data_publicacao'])) ?></small>
                        <p class="mt-2 text-gray-700"><?= nl2br(htmlspecialchars($aviso['conteudo'])) ?></p>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </section>

        <section class="bg-white p-6 rounded-xl shadow-md mb-6 text-center italic text-gray-600">
            <p>"<?= htmlspecialchars($frase_selecionada) ?>"</p>
        </section>
    </main>

    <script>
        // Lógica para alternar a barra lateral em dispositivos móveis
        const menuToggle = document.getElementById('menu-toggle');
        const sidebar = document.getElementById('sidebar');
        const overlay = document.getElementById('overlay');

        function toggleSidebar() {
            sidebar.classList.toggle('active');
            overlay.classList.toggle('active');
        }

        menuToggle.addEventListener('click', toggleSidebar);
        overlay.addEventListener('click', toggleSidebar);

        // A LÓGICA DE toggleReplyForm FOI REMOVIDA, POIS NÃO HÁ MAIS CHAT

        // Gráfico de Barras: Evolução de Notas por Turma do Professor (Dinâmico)
        const ctxNotasProfessor = document.getElementById('graficoNotasProfessor');
        new Chart(ctxNotasProfessor, {
            type: 'bar',
            data: {
                labels: <?= json_encode($labels_notas_turmas) ?>,
                datasets: <?= json_encode($datasets_notas_turmas) ?>
            },
            options: {
                responsive: true,
                maintainAspectRatio: false, // Permite que o gráfico use a altura definida pelo CSS
                plugins: {
                    legend: { position: 'top' }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        max: 20, // Assumindo notas de 0 a 20
                        title: { display: true, text: 'Média das Notas' }
                    }
                }
            }
        });
    </script>
</body>
</html>